!======================================================================|
! Calculation of the empty lattice band structure for a given sequence |
! of segments inside the 1st BZ                                        |
!======================================================================|

program al_mptl
  implicit none

  integer :: i,j,k
  ! start/end points of the segments
  double precision,allocatable,dimension(:,:)::kpoints
  ! a matrix of the eigenvalues on a single segment: each row stores the
  ! eigenvalues for a single k point along the segment  
  double precision,allocatable,dimension(:,:)::branch,tmp_mem
  double precision::pi=4.d0*datan(1.d0)  ! pi
  ! total number of high symmetry points in input (aka start/end
  ! points of the segments)
  double precision,allocatable,dimension(:) :: npoints
  integer :: nseg     ! n. of segments
  logical :: closure  ! whether to include the last point of a segment

!===================|
! INPUT STARTS HERE |
!===================|
  ! QE's alat (in angstrom) (eg lattice constant of the conventional fcc cell)
  double precision :: alat=7.6333*0.52917721 ! bohr -> angstrom

  ! reciprocal lattice vectors: cart. coord. in units of 2\pi/alat
  ! (copy&paste from QE's output)  
  double precision, dimension(3,3) :: brec=reshape( &
       (/-1.000000,-1.000000,1.000000,              &
       1.000000,1.000000,1.000000,                  &
       -1.000000,1.000000,-1.000000/),              &
       shape(brec),order=(/2,1/))

  ! (multi)segment trajectory: in QE's format;
  ! cart. coords. in units of 2\pi/alat
  ! (copy&paste from QE's output)  
  double precision :: trajectory(6,4)=reshape( &
       (/ &
       0.0   ,0.0  ,0.0   ,25. ,& ! gamma
       0.0   ,1.0  ,0.0   ,25. ,& ! X
       0.5   ,1.0  ,0.0   ,25. ,& ! W
       0.0   ,0.0  ,0.0   ,25. ,& ! gamma
       0.25  ,1.0  ,0.25  ,25. ,& ! U
       0.0   ,1.0  ,0.0   ,0.   & ! X
       /),shape(trajectory),order=(/2,1/))

  ! limits for generating G vectors:
  !        G=n1*b1+n2*b2+n3*b3
  ! n1,n2,n3\in[nmin,nmax]  
  integer :: nmin=-1,nmax=1

  ! how many eigenvalues to print for each k point
  integer :: neig=5

!=================|
! INPUT ENDS HERE |
!=================|

  

  brec=(2*pi/alat)*brec                   ! convert to ang^-1
  allocate(kpoints(size(trajectory,1),3))
  kpoints=(2*pi/alat)*trajectory(:,1:3)   ! ditto
  allocate(npoints(size(kpoints,1)))
  npoints=trajectory(:,4) ! pick npoints
  nseg=size(npoints)-1    ! n start/end points --> n-1 segments

  ! (nmax-nmin+1)**3 is the total n. of eigen per k point
  ! pick the min
  neig=min(neig,(nmax-nmin+1)**3)

  ! and that's it: loop over the segments
  do i=1,nseg
     closure=.false.
     if(i.eq.nseg)then ! raise closure for last segment
        closure=.true.
     endif
     ! eigenvalues for this segment
     branch=compute_ksegment(kpoints(i,:),kpoints(i+1,:),&
          npoints(i),closure,brec,nmin,nmax)
     ! dump to stdout
     do j=1,size(branch,1)
        print *, (branch(j,k),k=1,neig)
     enddo
     deallocate(branch)
  enddo

  ! that's all folks

contains

!============|
! SUBROUTINE |
!============|

! GB: stolen here: https://www.mjr19.org.uk/IT/sorts/sorts.f90
! sort array elements in ascending order (selection sort algo)

  subroutine inline_selection_sort(array)
    ! This version does not use Fortran's minloc intrinsic
    double precision, intent(inout) :: array(:)
    double precision :: temp
    integer :: i,j,k

    do i=1,size(array)-1
       temp=array(i)
       j=i
       do k=i+1,size(array)
          if (array(k).lt.temp) then
             temp=array(k)
             j=k
          endif
       enddo
       array(j)=array(i)
       array(i)=temp
    enddo

  end subroutine inline_selection_sort

!==========|
! FUNCTION |
!==========|

! Compute the MPTL eigenvalues for k points uniformly distributed along
! a given segment in the 1st BZ
! 
! INPUT
! K1        start point of the segment (cartesian coords)
! K2        end point of the segment (cartesian coords)
! nk        how many k points along the segment
! closure   whether to include K2 in the calculation
! brec      3x3 matrix whose rows are the cart.coords. of the
!           rec. latt. vectors
! nmin,nmax limits for generating G vectors:
!                  G=n1*b1+n2*b2+n3*b3
!           n1,n2,n3\in[nmin,nmax]
! 
! OUTPUT
! branch   nk[+1, when closure.eq..TRUE.]xNEIG matrix holding the
!          eigenvalues; NEIG is the total n. of eigenvalues per k point

  function compute_ksegment(K1,K2,nk,closure,brec,nmin,nmax) result(branch)
    implicit none
    double precision, intent(in) :: K1(3),K2(3)
    double precision, intent(in) :: nk
    logical, intent(in) :: closure
    double precision, intent(in) :: brec(3,3)
    integer, intent(in) :: nmin, nmax
    double precision, allocatable :: branch(:,:), tmp_mem(:,:)
    double precision :: lambda, delta, kp(3)
    integer :: nnk,i,j
    double precision, allocatable :: eigval(:)


    delta=1.d0/nk

    nnk=nk
    if (closure) then
       nnk=nk+1
    endif

    lambda=0.d0
    do i=1,nnk
       kp=K1+lambda*(K2-K1) ! generate current k point along K1--K2
       ! compute eigenvalues for this k point
       eigval=eigenvalue(kp,brec,nmin,nmax)
       ! (re)allocate branch
       if(.not.allocated(branch)) then
          allocate(branch(i,size(eigval)))
          call inline_selection_sort(eigval) ! sort the eigenvalues
          branch(i,:)=eigval
          deallocate(eigval)
       else
          call move_alloc(branch,tmp_mem)
          allocate(branch(i,size(eigval)))
          branch(1:i-1,:)=tmp_mem
          deallocate(tmp_mem)
          call inline_selection_sort(eigval) ! sort the eigenvalues
          branch(i,:)=eigval
          deallocate(eigval)
       endif
       lambda=lambda+delta ! next k point
    enddo

  end function compute_ksegment

!==========|
! FUNCTION |
!==========|

! Compute the MPTL eigenvalues for a single k point
! 
! INPUT
! kpoint    the k point
! brec      3x3 matrix whose rows are the cart.coords. of the
!           rec. latt. vectors
! nmin,nmax limits for generating G vectors:
!                  G=n1*b1+n2*b2+n3*b3
!           n1,n2,n3\in[nmin,nmax]
! 
! OUTPUT
! nrg_val  an allocated array holding the eigenvalues for kpoint,
!          (sorted in ascending order

  function eigenvalue(kpoint, brec, nmin, nmax) result(nrg_val)
    implicit none
    double precision, intent(in) :: kpoint(3)
    double precision, intent(in) :: brec(3,3)
    integer, intent(in) :: nmin, nmax

    double precision, allocatable :: nrg_val(:), tmp_mem(:)

    double precision :: factor=3.8099821 ! hbar^2/(2m) in eV*angstrom^2
    integer :: l ! counts the eigenvalues for this kpoint
    integer :: i,j,k
    double precision :: G(3)  ! G vector
    double precision :: modulus

    l=1 ! counts the eigenvalues (= G vectors)
    do i=nmin,nmax
       do j=nmin,nmax
          do k=nmin,nmax
             ! current G vector
             G=dble(i)*brec(1,:)+dble(j)*brec(2,:)+dble(k)*brec(3,:)
             modulus=sqrt(dot_product(kpoint-G,kpoint-G))
             ! (re)allocate nrg_val
             if(.not.allocated(nrg_val)) then
                allocate(nrg_val(l)) ! initial allocation (l=1)
                nrg_val(l)=factor*modulus*modulus
             else
                ! this is how reallocation works in fortran
                call move_alloc(nrg_val,tmp_mem)
                allocate(nrg_val(l))
                nrg_val(1:l-1)=tmp_mem(1:l-1)
                deallocate(tmp_mem)
                nrg_val(l)=factor*modulus*modulus
             endif
             l=l+1 ! next G vector
          enddo
       enddo
    enddo

  end function eigenvalue

end program al_mptl
